/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.Collections;

using Borland.Eco.Subscription;
using Borland.Eco.Interfaces;

namespace Borland.Eco.Subscription
{
	public abstract class SubscriberAdapterBase: ISubscriber
	{
		private WeakReference actualSubscriberWeakReference;

		protected abstract void DoReceive(object sender, EventArgs e, object actualSubscriber);

		protected SubscriberAdapterBase(object subscriber)
		{
			actualSubscriberWeakReference = new WeakReference(subscriber);
		}

		bool ISubscriber.Receive(System.Object sender, EventArgs e)
		{
			object actualSubscriber =  null;
			if (actualSubscriberWeakReference != null)
				actualSubscriber = actualSubscriberWeakReference.Target;
			if (actualSubscriber != null)
				DoReceive(sender, e, actualSubscriber);
			return (actualSubscriber != null);
		}

		public object ActualSubscriber
		{
			get
			{
				if (actualSubscriberWeakReference != null)
					return actualSubscriberWeakReference.Target;
				else
					return null;
			}
		}

		public void Deactivate()
		{
			actualSubscriberWeakReference = null;
		}

		public bool IsAlive()
		{
			return (actualSubscriberWeakReference != null) && (actualSubscriberWeakReference.IsAlive);
		}
	}


	// Subscriber for testing, saves all recieved events.
	// Deactivate() must be called after use to free resources, since it will otherwise be kept alive
	// as long as Pulisher lives.
	public sealed class TestSubscriber: ISubscriber
	{

		private ArrayList m_Events = new ArrayList();
        private bool m_Active = true;

		public EventArgs[] Events
		{
			get
			{
				EventArgs[] result = Array.CreateInstance(typeof(EventArgs), m_Events.Count) as  EventArgs[];
				m_Events.CopyTo(result);
				return result;
			}
		}

		public TestSubscriber(object subscriber)
		{
		}

		bool ISubscriber.Receive(System.Object sender, EventArgs e)
		{
			if (m_Active)
				m_Events.Add(e);
			return m_Active;
		}

		public void Clear()
		{
			m_Events = new ArrayList();
		}
		
		public void Deactivate()
		{
			m_Events = null;
			m_Active = false;;
		}

		bool ISubscriber.IsAlive()
		{
			return m_Active;
		}
	}


	public sealed class Publisher
	{
		public Publisher()
		{
		}

		private ISubscriber[] m_List = new ISubscriber[4];
		private int m_ShrinkLimit = -1;
		private int count;
		private Hashtable hashtable;
		private const int useHashtableCount = 10;

		private void AdjustSize(int newCount)
		{
			if (newCount == 0)
			{
				m_List = null;
				m_ShrinkLimit = -1;
				return;
			}
			if ((newCount >= m_ShrinkLimit) && (m_List != null) && (m_List.Length >= newCount))
				return;
			int newSize = 2 * newCount;
			ISubscriber[] newlist = new ISubscriber[newSize];
			if (m_List != null)
				Array.Copy(m_List, m_List.GetLowerBound(0), newlist, newlist.GetLowerBound(0), count);
			m_List = newlist;
			m_ShrinkLimit = newSize/2 - 2;
		}

		///<exception cref="ArgumentNullException">Thrown if <paramref name="subscriber"/> is null</exception>
		public void AddSubscription(ISubscriber subscriber)
		{
			if (subscriber == null)
				throw new ArgumentNullException("subscriber"); // do not localize
			if (ContainsSubscriber(subscriber))
				return;
			// Pack before growing, Assures that list doesn't grow to much if subscriptions
			// are placed and cancelled, and event never happens.
			AdjustSize(count + 1);
			m_List[count++] = subscriber;
			if (hashtable != null)
				hashtable.Add(subscriber, null);
		}

		public bool ContainsSubscriber(ISubscriber subscriber)
		{
			if ((hashtable == null) && (count > useHashtableCount))
			{
				hashtable = new Hashtable();
				for (int i = count; i-- > 0;)
					hashtable.Add(m_List[i], null);
			}
			if (hashtable == null)
			{
				for (int i = count; i-- > 0;)
					if (subscriber == m_List[i])
						return true;
				return false;
			}
			else
				return hashtable.ContainsKey(subscriber);
		}

		public void CollectSubscribers()
		{
			int RemovedEntries = 0;
			ISubscriber entry;

			for (int i = 0; i < count; i++)
			{
				entry = m_List[i];
				if (entry == null)
					throw new InvalidOperationException(InterfacesStringRes.sListIsNull);
				if (!entry.IsAlive())
				{
					RemovedEntries++;
					if (hashtable != null)
						hashtable.Remove(entry);
				}
				else
					if (RemovedEntries > 0)
					{
						m_List[i - RemovedEntries] = entry;
						m_List[i] = null;
					}
			}
			count -= RemovedEntries;
			if (count < useHashtableCount)
				hashtable = null;
		}

		public bool Send(System.Object sender, EventArgs e)
		{
			if (!HasSubscribers())
				return false;
			bool needsPacking = false;
			// must make a copy, in case new subscriptions are placed while sending.
			ISubscriber[] currentList = new ISubscriber[count];
			Array.Copy(m_List, m_List.GetLowerBound(0), currentList, currentList.GetLowerBound(0), count);
			int currentCount = count;
			for (int i = 0; i < currentCount; i++)
				if (!currentList[i].Receive(sender, e))
					needsPacking = true;
			if (needsPacking)
				CollectSubscribers();
			AdjustSize(count);
			return HasSubscribers();
		}

		public bool HasSubscribers()
		{
			return count > 0;
		}
	}

	public sealed class QualifiedPublisher
	{
		public QualifiedPublisher()
		{
		}

		private Hashtable hashtable = new Hashtable();

		private	Publisher EnsuredPublisherForQualifier(object qualifier)
		{
			Publisher p = (Publisher)hashtable[qualifier];
			if (p == null)
			{
				p = new Publisher();
				hashtable[qualifier] = p;
			}
			return p;
		}

		public bool HasPublisherForQualifier(object qualifier)
		{
			return hashtable.ContainsKey(qualifier);
		}

		public void AddSubscription(object qualifier, ISubscriber subscriber)
		{
			EnsuredPublisherForQualifier(qualifier).AddSubscription(subscriber);
		}

		public bool Send(object qualifier, object sender, EventArgs e)
		{
			bool HasSubscribers = false;
			Publisher p = (Publisher)hashtable[qualifier];

			if (p != null)
			{
				HasSubscribers = p.Send(sender, e);
				if (!HasSubscribers)
					hashtable.Remove(qualifier);
			}
			return HasSubscribers;
		}

	}

}

